<?php
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../includes/auth.php';

if (session_status() === PHP_SESSION_NONE) { session_name(SESSION_NAME); session_start(); }
if (!isLoggedIn()) { header('Location: ' . APP_URL . '/login.php'); exit; }

$isEdit = isset($_GET['id']);
$db = getDB();
$server = null;
$errors = [];

if ($isEdit) {
    $stmt = $db->prepare("SELECT * FROM servers WHERE id = ?");
    $stmt->execute([$_GET['id']]);
    $server = $stmt->fetch();
    if (!$server) { setFlash('error', 'Server nije pronađen.'); header('Location: index.php'); exit; }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRF($_POST['csrf_token'] ?? '')) { $errors[] = 'Nevažeći CSRF token.'; }
    
    $data = [
        'name' => trim($_POST['name'] ?? ''),
        'ip_address' => trim($_POST['ip_address'] ?? ''),
        'hostname' => trim($_POST['hostname'] ?? ''),
        'location' => trim($_POST['location'] ?? ''),
        'cpu' => trim($_POST['cpu'] ?? ''),
        'ram_gb' => (int)($_POST['ram_gb'] ?? 0),
        'storage_info' => trim($_POST['storage_info'] ?? ''),
        'os_version' => trim($_POST['os_version'] ?? ''),
        'proxmox_version' => trim($_POST['proxmox_version'] ?? ''),
        'install_date' => $_POST['install_date'] ?: null,
        'admin_username' => trim($_POST['admin_username'] ?? ''),
        'admin_password' => trim($_POST['admin_password'] ?? ''),
        'status' => $_POST['status'] ?? 'active',
        'notes' => trim($_POST['notes'] ?? '')
    ];
    
    if (empty($data['name'])) $errors[] = 'Naziv je obavezan.';
    if (empty($data['ip_address'])) $errors[] = 'IP adresa je obavezna.';
    if ($data['ip_address'] && !filter_var($data['ip_address'], FILTER_VALIDATE_IP)) $errors[] = 'IP adresa nije ispravna.';
    
    if (empty($errors)) {
        try {
            if ($isEdit) {
                $stmt = $db->prepare("UPDATE servers SET name=?, ip_address=?, hostname=?, location=?, cpu=?, ram_gb=?, storage_info=?, os_version=?, proxmox_version=?, install_date=?, admin_username=?, admin_password=?, status=?, notes=? WHERE id=?");
                $stmt->execute([$data['name'], $data['ip_address'], $data['hostname'], $data['location'], $data['cpu'], $data['ram_gb'], $data['storage_info'], $data['os_version'], $data['proxmox_version'], $data['install_date'], $data['admin_username'], $data['admin_password'], $data['status'], $data['notes'], $_GET['id']]);
                logChange('update', 'server', $_GET['id'], $data['name'], $server, $data);
                setFlash('success', 'Server je ažuriran.');
            } else {
                $stmt = $db->prepare("INSERT INTO servers (name, ip_address, hostname, location, cpu, ram_gb, storage_info, os_version, proxmox_version, install_date, admin_username, admin_password, status, notes) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$data['name'], $data['ip_address'], $data['hostname'], $data['location'], $data['cpu'], $data['ram_gb'], $data['storage_info'], $data['os_version'], $data['proxmox_version'], $data['install_date'], $data['admin_username'], $data['admin_password'], $data['status'], $data['notes']]);
                logChange('create', 'server', $db->lastInsertId(), $data['name'], null, $data);
                setFlash('success', 'Server je dodan.');
            }
            header('Location: index.php'); exit;
        } catch (PDOException $e) { $errors[] = 'Greška: ' . $e->getMessage(); }
    }
    $server = $data;
}

$pageTitle = ($isEdit ? 'Uredi' : 'Novi') . ' server - ' . APP_NAME;
$currentPage = 'servers';
require_once __DIR__ . '/../../includes/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h4><i class="bi bi-server me-2"></i><?= $isEdit ? 'Uredi: ' . clean($server['name']) : 'Novi server' ?></h4>
    <a href="index.php" class="btn btn-outline-secondary"><i class="bi bi-arrow-left me-1"></i>Natrag</a>
</div>

<?php if ($errors): ?><div class="alert alert-danger"><ul class="mb-0"><?php foreach ($errors as $e): ?><li><?= clean($e) ?></li><?php endforeach; ?></ul></div><?php endif; ?>

<div class="card">
    <div class="card-body">
        <form method="POST">
            <input type="hidden" name="csrf_token" value="<?= generateCSRF() ?>">
            <div class="row">
                <div class="col-md-6">
                    <h6 class="text-muted mb-3">Osnovni podaci</h6>
                    <div class="mb-3">
                        <label class="form-label required">Naziv servera</label>
                        <input type="text" name="name" class="form-control" value="<?= clean($server['name'] ?? '') ?>" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label required">IP adresa</label>
                        <input type="text" name="ip_address" class="form-control" value="<?= clean($server['ip_address'] ?? '') ?>" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Hostname</label>
                        <input type="text" name="hostname" class="form-control" value="<?= clean($server['hostname'] ?? '') ?>">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Lokacija</label>
                        <input type="text" name="location" class="form-control" value="<?= clean($server['location'] ?? '') ?>">
                    </div>
                    <div class="row">
                        <div class="col-6 mb-3">
                            <label class="form-label">Datum instalacije</label>
                            <input type="date" name="install_date" class="form-control" value="<?= $server['install_date'] ?? '' ?>">
                        </div>
                        <div class="col-6 mb-3">
                            <label class="form-label">Status</label>
                            <select name="status" class="form-select">
                                <option value="active" <?= ($server['status'] ?? '') === 'active' ? 'selected' : '' ?>>Active</option>
                                <option value="maintenance" <?= ($server['status'] ?? '') === 'maintenance' ? 'selected' : '' ?>>Maintenance</option>
                                <option value="offline" <?= ($server['status'] ?? '') === 'offline' ? 'selected' : '' ?>>Offline</option>
                            </select>
                        </div>
                    </div>
                    
                    <h6 class="text-muted mb-3 mt-4">Pristupni podaci</h6>
                    <div class="mb-3">
                        <label class="form-label">Admin username</label>
                        <input type="text" name="admin_username" class="form-control" value="<?= clean($server['admin_username'] ?? '') ?>" autocomplete="off">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Admin password</label>
                        <div class="input-group">
                            <input type="text" name="admin_password" class="form-control font-mono" value="<?= clean($server['admin_password'] ?? '') ?>" autocomplete="off">
                            <button type="button" class="btn btn-outline-secondary" onclick="generatePassword(this)"><i class="bi bi-key"></i></button>
                            <button type="button" class="btn btn-outline-secondary" onclick="copyPassword(this)"><i class="bi bi-clipboard"></i></button>
                        </div>
                    </div>
                </div>
                <div class="col-md-6">
                    <h6 class="text-muted mb-3">Specifikacije</h6>
                    <div class="mb-3">
                        <label class="form-label">CPU</label>
                        <input type="text" name="cpu" class="form-control" value="<?= clean($server['cpu'] ?? '') ?>" placeholder="Intel Xeon E5-2680 v4">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">RAM (GB)</label>
                        <input type="number" name="ram_gb" class="form-control" value="<?= (int)($server['ram_gb'] ?? 0) ?>" min="0">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Storage info</label>
                        <textarea name="storage_info" class="form-control" rows="2"><?= clean($server['storage_info'] ?? '') ?></textarea>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">OS verzija</label>
                        <input type="text" name="os_version" class="form-control" value="<?= clean($server['os_version'] ?? '') ?>">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Proxmox verzija</label>
                        <input type="text" name="proxmox_version" class="form-control" value="<?= clean($server['proxmox_version'] ?? '') ?>">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Napomene</label>
                        <textarea name="notes" class="form-control" rows="3"><?= clean($server['notes'] ?? '') ?></textarea>
                    </div>
                </div>
            </div>
            <div class="d-flex gap-2">
                <button type="submit" class="btn btn-primary"><i class="bi bi-check-lg me-1"></i><?= $isEdit ? 'Spremi' : 'Dodaj' ?></button>
                <a href="index.php" class="btn btn-outline-secondary">Odustani</a>
            </div>
        </form>
    </div>
</div>

<script>
function generatePassword(btn) {
    const chars = 'ABCDEFGHJKLMNPQRSTUVWXYZabcdefghjkmnpqrstuvwxyz23456789!@#$%&*';
    let pass = '';
    for (let i = 0; i < 16; i++) pass += chars.charAt(Math.floor(Math.random() * chars.length));
    btn.closest('.input-group').querySelector('input').value = pass;
}
function copyPassword(btn) {
    const input = btn.closest('.input-group').querySelector('input');
    navigator.clipboard.writeText(input.value);
    btn.innerHTML = '<i class="bi bi-check"></i>';
    setTimeout(() => btn.innerHTML = '<i class="bi bi-clipboard"></i>', 1500);
}
</script>

<?php require_once __DIR__ . '/../../includes/footer.php'; ?>
