<?php
/**
 * IT Inventory - Helper Functions
 */

require_once __DIR__ . '/../config.php';

// Sanitizacija inputa - null-safe
function clean($data): string {
    if ($data === null) return '';
    return htmlspecialchars(trim((string)$data), ENT_QUOTES, 'UTF-8');
}

// CSRF token
function generateCSRF(): string {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function verifyCSRF($token): bool {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Flash poruke
function setFlash(string $type, string $message): void {
    $_SESSION['flash'] = ['type' => $type, 'message' => $message];
}

function getFlash(): ?array {
    if (isset($_SESSION['flash'])) {
        $flash = $_SESSION['flash'];
        unset($_SESSION['flash']);
        return $flash;
    }
    return null;
}

// Provjera autentikacije
function isLoggedIn(): bool {
    return isset($_SESSION['admin_id']);
}

function requireLogin(): void {
    if (!isLoggedIn()) {
        header('Location: ' . APP_URL . '/login.php');
        exit;
    }
}

function isAdmin(): bool {
    return isset($_SESSION['admin_role']) && $_SESSION['admin_role'] === 'admin';
}

// Logiranje promjena - bez admin_name kolone
function logChange(string $action, string $entityType, int $entityId, string $entityName, ?array $oldValues = null, ?array $newValues = null): void {
    try {
        $db = getDB();
        $stmt = $db->prepare("
            INSERT INTO changelog (admin_id, action, entity_type, entity_id, entity_name, old_values, new_values, ip_address)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $_SESSION['admin_id'] ?? null,
            $action,
            $entityType,
            $entityId,
            $entityName,
            $oldValues ? json_encode($oldValues) : null,
            $newValues ? json_encode($newValues) : null,
            $_SERVER['REMOTE_ADDR'] ?? null
        ]);
    } catch (PDOException $e) {
        // Ignoriraj greške pri loggiranju da ne blokira glavnu operaciju
    }
}

// Paginacija
function paginate(int $total, int $perPage = 20, int $currentPage = 1): array {
    $totalPages = ceil($total / $perPage);
    $currentPage = max(1, min($currentPage, $totalPages));
    $offset = ($currentPage - 1) * $perPage;
    
    return [
        'total' => $total,
        'per_page' => $perPage,
        'current_page' => $currentPage,
        'total_pages' => $totalPages,
        'offset' => $offset,
        'has_prev' => $currentPage > 1,
        'has_next' => $currentPage < $totalPages
    ];
}

// Status badge
function statusBadge(?string $status): string {
    if (!$status) return '<span class="badge bg-secondary">-</span>';
    $classes = [
        'active' => 'bg-success',
        'running' => 'bg-success',
        'maintenance' => 'bg-warning',
        'offline' => 'bg-danger',
        'stopped' => 'bg-secondary',
        'disabled' => 'bg-secondary',
        'suspended' => 'bg-warning',
        'retired' => 'bg-dark',
        'storage' => 'bg-info',
        'template' => 'bg-info'
    ];
    $class = $classes[$status] ?? 'bg-secondary';
    return "<span class=\"badge {$class}\">" . ucfirst($status) . "</span>";
}

// Formatiranje veličine
function formatSize($gb): string {
    if (!$gb) return '-';
    if ($gb >= 1024) {
        return round($gb / 1024, 2) . ' TB';
    }
    return $gb . ' GB';
}

// Formatiranje datuma
function formatDate($date): string {
    if (!$date) return '-';
    return date('d.m.Y.', strtotime($date));
}

function formatDateTime($datetime): string {
    if (!$datetime) return '-';
    return date('d.m.Y. H:i', strtotime($datetime));
}