<?php
/**
 * IT Inventory - Add/Edit Server
 */
$isEdit = isset($_GET['id']);
$pageTitle = ($isEdit ? 'Uredi' : 'Novi') . ' server - IT Inventory';
$currentPage = 'servers';

require_once __DIR__ . '/../../includes/header.php';
requireLogin();

$db = getDB();
$server = null;
$errors = [];

// Ako je edit, dohvati postojeći server
if ($isEdit) {
    $stmt = $db->prepare("SELECT * FROM servers WHERE id = ?");
    $stmt->execute([$_GET['id']]);
    $server = $stmt->fetch();
    
    if (!$server) {
        setFlash('error', 'Server nije pronađen.');
        header('Location: index.php');
        exit;
    }
}

// Obrada forme
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRF($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Nevažeći CSRF token.';
    }
    
    $data = [
        'name' => trim($_POST['name'] ?? ''),
        'ip_address' => trim($_POST['ip_address'] ?? ''),
        'hostname' => trim($_POST['hostname'] ?? ''),
        'location' => trim($_POST['location'] ?? ''),
        'cpu' => trim($_POST['cpu'] ?? ''),
        'ram_gb' => (int)($_POST['ram_gb'] ?? 0),
        'storage_info' => trim($_POST['storage_info'] ?? ''),
        'os_version' => trim($_POST['os_version'] ?? ''),
        'proxmox_version' => trim($_POST['proxmox_version'] ?? ''),
        'status' => $_POST['status'] ?? 'active',
        'notes' => trim($_POST['notes'] ?? '')
    ];
    
    // Validacija
    if (empty($data['name'])) $errors[] = 'Naziv je obavezan.';
    if (empty($data['ip_address'])) $errors[] = 'IP adresa je obavezna.';
    if (!filter_var($data['ip_address'], FILTER_VALIDATE_IP)) $errors[] = 'IP adresa nije ispravna.';
    
    if (empty($errors)) {
        try {
            if ($isEdit) {
                $oldValues = $server;
                
                $stmt = $db->prepare("
                    UPDATE servers SET 
                        name = ?, ip_address = ?, hostname = ?, location = ?, cpu = ?,
                        ram_gb = ?, storage_info = ?, os_version = ?, proxmox_version = ?,
                        status = ?, notes = ?
                    WHERE id = ?
                ");
                $stmt->execute([
                    $data['name'], $data['ip_address'], $data['hostname'], $data['location'],
                    $data['cpu'], $data['ram_gb'], $data['storage_info'], $data['os_version'],
                    $data['proxmox_version'], $data['status'], $data['notes'], $_GET['id']
                ]);
                
                logChange('update', 'server', $_GET['id'], $data['name'], $oldValues, $data);
                setFlash('success', 'Server je uspješno ažuriran.');
            } else {
                $stmt = $db->prepare("
                    INSERT INTO servers (name, ip_address, hostname, location, cpu, ram_gb, storage_info, os_version, proxmox_version, status, notes)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([
                    $data['name'], $data['ip_address'], $data['hostname'], $data['location'],
                    $data['cpu'], $data['ram_gb'], $data['storage_info'], $data['os_version'],
                    $data['proxmox_version'], $data['status'], $data['notes']
                ]);
                
                $newId = $db->lastInsertId();
                logChange('create', 'server', $newId, $data['name'], null, $data);
                setFlash('success', 'Server je uspješno dodan.');
            }
            
            header('Location: index.php');
            exit;
        } catch (PDOException $e) {
            $errors[] = 'Greška pri spremanju: ' . $e->getMessage();
        }
    }
    
    // Ako ima grešaka, zadrži unesene podatke
    $server = $data;
    if ($isEdit) $server['id'] = $_GET['id'];
}
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h4>
        <i class="bi bi-server me-2"></i>
        <?= $isEdit ? 'Uredi server: ' . clean($server['name']) : 'Novi server' ?>
    </h4>
    <a href="index.php" class="btn btn-outline-secondary">
        <i class="bi bi-arrow-left me-1"></i>Natrag
    </a>
</div>

<?php if ($errors): ?>
<div class="alert alert-danger">
    <ul class="mb-0">
        <?php foreach ($errors as $error): ?>
        <li><?= clean($error) ?></li>
        <?php endforeach; ?>
    </ul>
</div>
<?php endif; ?>

<div class="card">
    <div class="card-body">
        <form method="POST">
            <input type="hidden" name="csrf_token" value="<?= generateCSRF() ?>">
            
            <div class="row">
                <div class="col-md-6">
                    <h6 class="text-muted mb-3">Osnovni podaci</h6>
                    
                    <div class="mb-3">
                        <label class="form-label required">Naziv servera</label>
                        <input type="text" name="name" class="form-control" value="<?= clean($server['name'] ?? '') ?>" required>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label required">IP adresa</label>
                        <input type="text" name="ip_address" class="form-control" value="<?= clean($server['ip_address'] ?? '') ?>" placeholder="192.168.1.100" required>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Hostname</label>
                        <input type="text" name="hostname" class="form-control" value="<?= clean($server['hostname'] ?? '') ?>" placeholder="pve1.local">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Lokacija</label>
                        <input type="text" name="location" class="form-control" value="<?= clean($server['location'] ?? '') ?>" placeholder="Server soba, Rack 1">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Status</label>
                        <select name="status" class="form-select">
                            <option value="active" <?= ($server['status'] ?? '') === 'active' ? 'selected' : '' ?>>Active</option>
                            <option value="maintenance" <?= ($server['status'] ?? '') === 'maintenance' ? 'selected' : '' ?>>Maintenance</option>
                            <option value="offline" <?= ($server['status'] ?? '') === 'offline' ? 'selected' : '' ?>>Offline</option>
                        </select>
                    </div>
                </div>
                
                <div class="col-md-6">
                    <h6 class="text-muted mb-3">Specifikacije</h6>
                    
                    <div class="mb-3">
                        <label class="form-label">CPU</label>
                        <input type="text" name="cpu" class="form-control" value="<?= clean($server['cpu'] ?? '') ?>" placeholder="Intel Xeon E5-2680 v4">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">RAM (GB)</label>
                        <input type="number" name="ram_gb" class="form-control" value="<?= (int)($server['ram_gb'] ?? 0) ?>" min="0">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Storage info</label>
                        <textarea name="storage_info" class="form-control" rows="2" placeholder="2x 500GB SSD, 4x 2TB HDD"><?= clean($server['storage_info'] ?? '') ?></textarea>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">OS verzija</label>
                        <input type="text" name="os_version" class="form-control" value="<?= clean($server['os_version'] ?? '') ?>" placeholder="Debian 12">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Proxmox verzija</label>
                        <input type="text" name="proxmox_version" class="form-control" value="<?= clean($server['proxmox_version'] ?? '') ?>" placeholder="8.1">
                    </div>
                </div>
            </div>
            
            <div class="mb-3">
                <label class="form-label">Napomene</label>
                <textarea name="notes" class="form-control" rows="3"><?= clean($server['notes'] ?? '') ?></textarea>
            </div>
            
            <div class="d-flex gap-2">
                <button type="submit" class="btn btn-primary">
                    <i class="bi bi-check-lg me-1"></i><?= $isEdit ? 'Spremi promjene' : 'Dodaj server' ?>
                </button>
                <a href="index.php" class="btn btn-outline-secondary">Odustani</a>
            </div>
        </form>
    </div>
</div>

<?php require_once __DIR__ . '/../../includes/footer.php'; ?>
