<?php
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../includes/auth.php';

if (session_status() === PHP_SESSION_NONE) {
    session_name(SESSION_NAME);
    session_start();
}

if (!isLoggedIn()) {
    header('Location: ' . APP_URL . '/login.php');
    exit;
}

$isEdit = isset($_GET['id']);
$db = getDB();
$user = null;
$errors = [];

if ($isEdit) {
    $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_GET['id']]);
    $user = $stmt->fetch();
    if (!$user) { 
        setFlash('error', 'Korisnik nije pronađen.'); 
        header('Location: index.php'); 
        exit; 
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRF($_POST['csrf_token'] ?? '')) { 
        $errors[] = 'Nevažeći CSRF token.'; 
    }
    
    $data = [
        'username' => trim($_POST['username'] ?? ''),
        'full_name' => trim($_POST['full_name'] ?? ''),
        'email' => trim($_POST['email'] ?? ''),
        'department' => trim($_POST['department'] ?? ''),
        'position' => trim($_POST['position'] ?? ''),
        'phone' => trim($_POST['phone'] ?? ''),
        'private_phone' => trim($_POST['private_phone'] ?? ''),
        'work_phone' => trim($_POST['work_phone'] ?? ''),
        'ad_enabled' => isset($_POST['ad_enabled']) ? 1 : 0,
        'ad_password' => trim($_POST['ad_password'] ?? ''),
        'zimbra_enabled' => isset($_POST['zimbra_enabled']) ? 1 : 0,
        'zimbra_quota_gb' => (float)($_POST['zimbra_quota_gb'] ?? 0),
        'zimbra_password' => trim($_POST['zimbra_password'] ?? ''),
        'nextcloud_enabled' => isset($_POST['nextcloud_enabled']) ? 1 : 0,
        'nextcloud_quota_gb' => (float)($_POST['nextcloud_quota_gb'] ?? 0),
        'nextcloud_password' => trim($_POST['nextcloud_password'] ?? ''),
        'status' => $_POST['status'] ?? 'active',
        'notes' => trim($_POST['notes'] ?? '')
    ];
    
    if (empty($data['username'])) $errors[] = 'Username je obavezan.';
    if (empty($data['full_name'])) $errors[] = 'Ime i prezime je obavezno.';
    
    if (empty($errors)) {
        try {
            if ($isEdit) {
                $stmt = $db->prepare("UPDATE users SET username=?, full_name=?, email=?, department=?, position=?, phone=?, private_phone=?, work_phone=?, ad_enabled=?, ad_password=?, zimbra_enabled=?, zimbra_quota_gb=?, zimbra_password=?, nextcloud_enabled=?, nextcloud_quota_gb=?, nextcloud_password=?, status=?, notes=? WHERE id=?");
                $stmt->execute([$data['username'], $data['full_name'], $data['email'], $data['department'], $data['position'], $data['phone'], $data['private_phone'], $data['work_phone'], $data['ad_enabled'], $data['ad_password'], $data['zimbra_enabled'], $data['zimbra_quota_gb'], $data['zimbra_password'], $data['nextcloud_enabled'], $data['nextcloud_quota_gb'], $data['nextcloud_password'], $data['status'], $data['notes'], $_GET['id']]);
                logChange('update', 'user', $_GET['id'], $data['full_name'], $user, $data);
                setFlash('success', 'Korisnik je ažuriran.');
            } else {
                $stmt = $db->prepare("INSERT INTO users (username, full_name, email, department, position, phone, private_phone, work_phone, ad_enabled, ad_password, zimbra_enabled, zimbra_quota_gb, zimbra_password, nextcloud_enabled, nextcloud_quota_gb, nextcloud_password, status, notes) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$data['username'], $data['full_name'], $data['email'], $data['department'], $data['position'], $data['phone'], $data['private_phone'], $data['work_phone'], $data['ad_enabled'], $data['ad_password'], $data['zimbra_enabled'], $data['zimbra_quota_gb'], $data['zimbra_password'], $data['nextcloud_enabled'], $data['nextcloud_quota_gb'], $data['nextcloud_password'], $data['status'], $data['notes']]);
                logChange('create', 'user', $db->lastInsertId(), $data['full_name'], null, $data);
                setFlash('success', 'Korisnik je dodan.');
            }
            header('Location: index.php'); 
            exit;
        } catch (PDOException $e) { 
            $errors[] = 'Greška: ' . $e->getMessage(); 
        }
    }
    $user = $data;
}

$pageTitle = ($isEdit ? 'Uredi' : 'Novi') . ' korisnik - IT Inventory';
$currentPage = 'users';
require_once __DIR__ . '/../../includes/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h4><i class="bi bi-person me-2"></i><?= $isEdit ? 'Uredi: ' . clean($user['full_name']) : 'Novi korisnik' ?></h4>
    <a href="index.php" class="btn btn-outline-secondary"><i class="bi bi-arrow-left me-1"></i>Natrag</a>
</div>

<?php if ($errors): ?><div class="alert alert-danger"><ul class="mb-0"><?php foreach ($errors as $e): ?><li><?= clean($e) ?></li><?php endforeach; ?></ul></div><?php endif; ?>

<div class="card">
    <div class="card-body">
        <form method="POST">
            <input type="hidden" name="csrf_token" value="<?= generateCSRF() ?>">
            <div class="row">
                <div class="col-md-6">
                    <h6 class="text-muted mb-3">Osnovni podaci</h6>
                    <div class="mb-3">
                        <label class="form-label required">Username</label>
                        <input type="text" name="username" class="form-control" value="<?= clean($user['username'] ?? '') ?>" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label required">Ime i prezime</label>
                        <input type="text" name="full_name" class="form-control" value="<?= clean($user['full_name'] ?? '') ?>" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Email</label>
                        <input type="email" name="email" class="form-control" value="<?= clean($user['email'] ?? '') ?>">
                    </div>
                    <div class="row">
                        <div class="col-6 mb-3">
                            <label class="form-label">Odjel</label>
                            <input type="text" name="department" class="form-control" value="<?= clean($user['department'] ?? '') ?>">
                        </div>
                        <div class="col-6 mb-3">
                            <label class="form-label">Pozicija</label>
                            <input type="text" name="position" class="form-control" value="<?= clean($user['position'] ?? '') ?>">
                        </div>
                    </div>
                    
                    <h6 class="text-muted mb-3 mt-4">Kontakt brojevi</h6>
                    <div class="row">
                        <div class="col-4 mb-3">
                            <label class="form-label">Telefon (glavni)</label>
                            <input type="text" name="phone" class="form-control" value="<?= clean($user['phone'] ?? '') ?>">
                        </div>
                        <div class="col-4 mb-3">
                            <label class="form-label">Službeni mob.</label>
                            <input type="text" name="work_phone" class="form-control" value="<?= clean($user['work_phone'] ?? '') ?>">
                        </div>
                        <div class="col-4 mb-3">
                            <label class="form-label">Privatni mob.</label>
                            <input type="text" name="private_phone" class="form-control" value="<?= clean($user['private_phone'] ?? '') ?>">
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Status</label>
                        <select name="status" class="form-select">
                            <option value="active" <?= ($user['status'] ?? '') === 'active' ? 'selected' : '' ?>>Active</option>
                            <option value="disabled" <?= ($user['status'] ?? '') === 'disabled' ? 'selected' : '' ?>>Disabled</option>
                            <option value="suspended" <?= ($user['status'] ?? '') === 'suspended' ? 'selected' : '' ?>>Suspended</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Napomene</label>
                        <textarea name="notes" class="form-control" rows="3"><?= clean($user['notes'] ?? '') ?></textarea>
                    </div>
                </div>
                <div class="col-md-6">
                    <h6 class="text-muted mb-3">Servisi i pristupni podaci</h6>
                    
                    <div class="card mb-3">
                        <div class="card-body">
                            <div class="form-check mb-2">
                                <input type="checkbox" name="ad_enabled" class="form-check-input" id="ad" <?= ($user['ad_enabled'] ?? 0) ? 'checked' : '' ?>>
                                <label class="form-check-label" for="ad"><strong>Active Directory</strong></label>
                            </div>
                            <div class="ms-4">
                                <label class="form-label">Lozinka</label>
                                <div class="input-group">
                                    <input type="text" name="ad_password" class="form-control form-control-sm password-field" value="<?= clean($user['ad_password'] ?? '') ?>">
                                    <button type="button" class="btn btn-sm btn-outline-secondary generate-password" title="Generiraj lozinku"><i class="bi bi-shuffle"></i></button>
                                    <button type="button" class="btn btn-sm btn-outline-secondary copy-password" title="Kopiraj"><i class="bi bi-clipboard"></i></button>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="card mb-3">
                        <div class="card-body">
                            <div class="form-check mb-2">
                                <input type="checkbox" name="zimbra_enabled" class="form-check-input" id="zimbra" <?= ($user['zimbra_enabled'] ?? 0) ? 'checked' : '' ?>>
                                <label class="form-check-label" for="zimbra"><strong>Zimbra Mail</strong></label>
                            </div>
                            <div class="ms-4">
                                <div class="row">
                                    <div class="col-5 mb-2">
                                        <label class="form-label">Kvota (GB)</label>
                                        <input type="number" name="zimbra_quota_gb" class="form-control form-control-sm" value="<?= (float)($user['zimbra_quota_gb'] ?? 0) ?>" step="0.5" min="0">
                                    </div>
                                </div>
                                <label class="form-label">Lozinka</label>
                                <div class="input-group">
                                    <input type="text" name="zimbra_password" class="form-control form-control-sm password-field" value="<?= clean($user['zimbra_password'] ?? '') ?>">
                                    <button type="button" class="btn btn-sm btn-outline-secondary generate-password" title="Generiraj lozinku"><i class="bi bi-shuffle"></i></button>
                                    <button type="button" class="btn btn-sm btn-outline-secondary copy-password" title="Kopiraj"><i class="bi bi-clipboard"></i></button>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="card mb-3">
                        <div class="card-body">
                            <div class="form-check mb-2">
                                <input type="checkbox" name="nextcloud_enabled" class="form-check-input" id="nc" <?= ($user['nextcloud_enabled'] ?? 0) ? 'checked' : '' ?>>
                                <label class="form-check-label" for="nc"><strong>Nextcloud</strong></label>
                            </div>
                            <div class="ms-4">
                                <div class="row">
                                    <div class="col-5 mb-2">
                                        <label class="form-label">Kvota (GB)</label>
                                        <input type="number" name="nextcloud_quota_gb" class="form-control form-control-sm" value="<?= (float)($user['nextcloud_quota_gb'] ?? 0) ?>" step="0.5" min="0">
                                    </div>
                                </div>
                                <label class="form-label">Lozinka</label>
                                <div class="input-group">
                                    <input type="text" name="nextcloud_password" class="form-control form-control-sm password-field" value="<?= clean($user['nextcloud_password'] ?? '') ?>">
                                    <button type="button" class="btn btn-sm btn-outline-secondary generate-password" title="Generiraj lozinku"><i class="bi bi-shuffle"></i></button>
                                    <button type="button" class="btn btn-sm btn-outline-secondary copy-password" title="Kopiraj"><i class="bi bi-clipboard"></i></button>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="card bg-light">
                        <div class="card-body py-2">
                            <small class="text-muted">
                                <i class="bi bi-info-circle me-1"></i>
                                Klikni <i class="bi bi-shuffle"></i> za generiranje sigurne lozinke (12 znakova, velika/mala slova, brojevi, simboli)
                            </small>
                        </div>
                    </div>
                </div>
            </div>
            <div class="d-flex gap-2 mt-3">
                <button type="submit" class="btn btn-primary"><i class="bi bi-check-lg me-1"></i><?= $isEdit ? 'Spremi' : 'Dodaj' ?></button>
                <a href="index.php" class="btn btn-outline-secondary">Odustani</a>
            </div>
        </form>
    </div>
</div>

<script>
// Password generator
function generatePassword(length = 12) {
    const upper = 'ABCDEFGHJKLMNPQRSTUVWXYZ';
    const lower = 'abcdefghjkmnpqrstuvwxyz';
    const numbers = '23456789';
    const symbols = '!@#$%&*';
    const all = upper + lower + numbers + symbols;
    
    let password = '';
    password += upper[Math.floor(Math.random() * upper.length)];
    password += lower[Math.floor(Math.random() * lower.length)];
    password += numbers[Math.floor(Math.random() * numbers.length)];
    password += symbols[Math.floor(Math.random() * symbols.length)];
    
    for (let i = 4; i < length; i++) {
        password += all[Math.floor(Math.random() * all.length)];
    }
    
    return password.split('').sort(() => Math.random() - 0.5).join('');
}

document.querySelectorAll('.generate-password').forEach(btn => {
    btn.addEventListener('click', function() {
        const input = this.closest('.input-group').querySelector('.password-field');
        input.value = generatePassword();
        input.select();
    });
});

document.querySelectorAll('.copy-password').forEach(btn => {
    btn.addEventListener('click', function() {
        const input = this.closest('.input-group').querySelector('.password-field');
        navigator.clipboard.writeText(input.value);
        
        const icon = this.querySelector('i');
        icon.classList.remove('bi-clipboard');
        icon.classList.add('bi-check');
        setTimeout(() => {
            icon.classList.remove('bi-check');
            icon.classList.add('bi-clipboard');
        }, 1500);
    });
});
</script>

<?php require_once __DIR__ . '/../../includes/footer.php'; ?>
