<?php
$pageTitle = 'Mrežni dijagram - IT Inventory';
$currentPage = 'diagram';

require_once __DIR__ . '/includes/header.php';
requireLogin();

$db = getDB();

// Dohvati servere s virtualkama
$servers = $db->query("SELECT * FROM servers WHERE status != 'retired' ORDER BY name")->fetchAll();

// Dohvati virtualke za svaki server
$vmsByServer = [];
foreach ($servers as $server) {
    $stmt = $db->prepare("SELECT * FROM virtual_machines WHERE host_server_id = ? ORDER BY name");
    $stmt->execute([$server['id']]);
    $vmsByServer[$server['id']] = $stmt->fetchAll();
}

// Dohvati mrežnu opremu
$equipment = [];
try {
    $equipment = $db->query("SELECT * FROM equipment WHERE status = 'active' ORDER BY type, name")->fetchAll();
} catch (PDOException $e) {}

$typeIcons = [
    'switch' => 'bi-diagram-3',
    'router' => 'bi-router',
    'firewall' => 'bi-shield-lock',
    'ups' => 'bi-battery-charging',
    'storage' => 'bi-device-hdd',
    'nas' => 'bi-hdd-stack',
    'san' => 'bi-hdd-network',
    'ap' => 'bi-wifi',
    'other' => 'bi-box'
];
?>

<style>
.diagram-container {
    background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
    border-radius: 16px;
    padding: 2rem;
    min-height: 600px;
    overflow: auto;
}

.diagram-section {
    margin-bottom: 2rem;
}

.diagram-section-title {
    color: rgba(255,255,255,0.6);
    font-size: 0.75rem;
    text-transform: uppercase;
    letter-spacing: 0.1em;
    margin-bottom: 1rem;
    padding-left: 0.5rem;
    border-left: 3px solid #6366f1;
}

.server-node {
    background: linear-gradient(135deg, #2d3748 0%, #1a202c 100%);
    border: 2px solid #4a5568;
    border-radius: 12px;
    padding: 1.25rem;
    margin-bottom: 1rem;
    transition: all 0.3s ease;
    cursor: pointer;
}

.server-node:hover {
    border-color: #6366f1;
    transform: translateY(-2px);
    box-shadow: 0 10px 40px rgba(99, 102, 241, 0.3);
}

.server-node.expanded {
    border-color: #6366f1;
}

.server-header {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.server-icon {
    width: 48px;
    height: 48px;
    background: linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%);
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    color: white;
}

.server-info h5 {
    color: white;
    margin: 0;
    font-size: 1rem;
    font-weight: 600;
}

.server-info p {
    color: rgba(255,255,255,0.5);
    margin: 0;
    font-size: 0.8rem;
}

.server-stats {
    display: flex;
    gap: 1rem;
    margin-left: auto;
}

.server-stat {
    text-align: center;
    padding: 0.5rem 1rem;
    background: rgba(255,255,255,0.05);
    border-radius: 8px;
}

.server-stat-value {
    color: #10b981;
    font-weight: 700;
    font-size: 1.1rem;
}

.server-stat-label {
    color: rgba(255,255,255,0.4);
    font-size: 0.7rem;
    text-transform: uppercase;
}

.vm-list {
    margin-top: 1rem;
    padding-top: 1rem;
    border-top: 1px solid rgba(255,255,255,0.1);
    display: none;
}

.server-node.expanded .vm-list {
    display: block;
}

.vm-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
    gap: 0.75rem;
}

.vm-node {
    background: rgba(255,255,255,0.05);
    border: 1px solid rgba(255,255,255,0.1);
    border-radius: 8px;
    padding: 0.75rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
    transition: all 0.2s;
}

.vm-node:hover {
    background: rgba(255,255,255,0.1);
    border-color: rgba(255,255,255,0.2);
}

.vm-icon {
    width: 32px;
    height: 32px;
    background: linear-gradient(135deg, #06b6d4 0%, #0891b2 100%);
    border-radius: 6px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 0.9rem;
}

.vm-info h6 {
    color: white;
    margin: 0;
    font-size: 0.85rem;
}

.vm-info small {
    color: rgba(255,255,255,0.4);
    font-size: 0.7rem;
}

.vm-status {
    margin-left: auto;
    width: 8px;
    height: 8px;
    border-radius: 50%;
    background: #10b981;
}

.vm-status.stopped {
    background: #6b7280;
}

.vm-status.maintenance {
    background: #f59e0b;
}

/* Equipment */
.equipment-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(180px, 1fr));
    gap: 1rem;
}

.equipment-node {
    background: linear-gradient(135deg, #2d3748 0%, #1a202c 100%);
    border: 2px solid #4a5568;
    border-radius: 10px;
    padding: 1rem;
    text-align: center;
    transition: all 0.3s;
}

.equipment-node:hover {
    border-color: #10b981;
    transform: translateY(-2px);
}

.equipment-icon {
    width: 48px;
    height: 48px;
    margin: 0 auto 0.75rem;
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    color: white;
}

.equipment-node h6 {
    color: white;
    margin: 0 0 0.25rem;
    font-size: 0.9rem;
}

.equipment-node small {
    color: rgba(255,255,255,0.4);
    font-size: 0.75rem;
}

.equipment-node code {
    display: block;
    margin-top: 0.5rem;
    background: rgba(255,255,255,0.1);
    padding: 0.25rem 0.5rem;
    border-radius: 4px;
    font-size: 0.7rem;
    color: #06b6d4;
}

/* Connections visual */
.connection-line {
    position: absolute;
    background: linear-gradient(90deg, #6366f1, #06b6d4);
    height: 2px;
}

/* Toggle button */
.expand-btn {
    color: rgba(255,255,255,0.4);
    margin-left: 1rem;
}

.server-node.expanded .expand-btn i {
    transform: rotate(180deg);
}

/* Legend */
.diagram-legend {
    display: flex;
    gap: 1.5rem;
    padding: 1rem;
    background: rgba(255,255,255,0.05);
    border-radius: 8px;
    margin-bottom: 1.5rem;
}

.legend-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: rgba(255,255,255,0.6);
    font-size: 0.8rem;
}

.legend-dot {
    width: 10px;
    height: 10px;
    border-radius: 50%;
}

.legend-dot.running { background: #10b981; }
.legend-dot.stopped { background: #6b7280; }
.legend-dot.maintenance { background: #f59e0b; }

@media print {
    .diagram-container {
        background: white !important;
    }
    .server-node, .equipment-node {
        background: white !important;
        border: 1px solid #ddd !important;
    }
    .server-info h5, .vm-info h6, .equipment-node h6 {
        color: black !important;
    }
}
</style>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h4><i class="bi bi-diagram-3 me-2"></i>Mrežni dijagram</h4>
    <div>
        <button class="btn btn-outline-secondary" onclick="expandAll()"><i class="bi bi-arrows-expand me-1"></i>Proširi sve</button>
        <button class="btn btn-outline-secondary" onclick="collapseAll()"><i class="bi bi-arrows-collapse me-1"></i>Skupi sve</button>
        <button class="btn btn-outline-primary" onclick="window.print()"><i class="bi bi-printer me-1"></i>Ispiši</button>
    </div>
</div>

<div class="diagram-container">
    <div class="diagram-legend">
        <div class="legend-item"><div class="legend-dot running"></div> Running</div>
        <div class="legend-item"><div class="legend-dot stopped"></div> Stopped</div>
        <div class="legend-item"><div class="legend-dot maintenance"></div> Maintenance</div>
        <div class="legend-item"><i class="bi bi-server text-white me-1"></i> Fizički server</div>
        <div class="legend-item"><i class="bi bi-pc-display text-info me-1"></i> Virtualni server</div>
    </div>

    <?php if (!empty($equipment)): ?>
    <div class="diagram-section">
        <div class="diagram-section-title">Mrežna oprema</div>
        <div class="equipment-grid">
            <?php foreach ($equipment as $eq): ?>
            <a href="<?= APP_URL ?>/modules/equipment/view.php?id=<?= $eq['id'] ?>" class="equipment-node text-decoration-none">
                <div class="equipment-icon">
                    <i class="bi <?= $typeIcons[$eq['type']] ?? 'bi-box' ?>"></i>
                </div>
                <h6><?= clean($eq['name']) ?></h6>
                <small><?= clean($eq['manufacturer']) ?> <?= clean($eq['model']) ?></small>
                <?php if ($eq['ip_address']): ?>
                <code><?= clean($eq['ip_address']) ?></code>
                <?php endif; ?>
            </a>
            <?php endforeach; ?>
        </div>
    </div>
    <?php endif; ?>

    <div class="diagram-section">
        <div class="diagram-section-title">Serveri & Virtualne mašine</div>
        
        <?php foreach ($servers as $server): ?>
        <?php $vms = $vmsByServer[$server['id']] ?? []; ?>
        <div class="server-node" onclick="toggleServer(this)">
            <div class="server-header">
                <div class="server-icon">
                    <i class="bi bi-server"></i>
                </div>
                <div class="server-info">
                    <h5><?= clean($server['name']) ?></h5>
                    <p>
                        <?= clean($server['manufacturer']) ?> <?= clean($server['model']) ?>
                        <?php if ($server['ip_address']): ?> · <code><?= clean($server['ip_address']) ?></code><?php endif; ?>
                    </p>
                </div>
                <div class="server-stats">
                    <div class="server-stat">
                        <div class="server-stat-value"><?= $server['cpu_cores'] ?: '-' ?></div>
                        <div class="server-stat-label">CPU</div>
                    </div>
                    <div class="server-stat">
                        <div class="server-stat-value"><?= $server['ram_gb'] ?: '-' ?></div>
                        <div class="server-stat-label">RAM GB</div>
                    </div>
                    <div class="server-stat">
                        <div class="server-stat-value"><?= count($vms) ?></div>
                        <div class="server-stat-label">VM</div>
                    </div>
                </div>
                <span class="expand-btn"><i class="bi bi-chevron-down"></i></span>
            </div>
            
            <?php if (!empty($vms)): ?>
            <div class="vm-list">
                <div class="vm-grid">
                    <?php foreach ($vms as $vm): ?>
                    <a href="<?= APP_URL ?>/modules/vms/view.php?id=<?= $vm['id'] ?>" class="vm-node text-decoration-none" onclick="event.stopPropagation()">
                        <div class="vm-icon">
                            <i class="bi bi-pc-display"></i>
                        </div>
                        <div class="vm-info">
                            <h6><?= clean($vm['name']) ?></h6>
                            <small><?= $vm['vcpu'] ?> vCPU · <?= $vm['ram_gb'] ?> GB · <?= clean($vm['ip_address']) ?></small>
                        </div>
                        <div class="vm-status <?= $vm['status'] === 'running' ? '' : ($vm['status'] === 'maintenance' ? 'maintenance' : 'stopped') ?>"></div>
                    </a>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>
        </div>
        <?php endforeach; ?>
        
        <?php if (empty($servers)): ?>
        <div class="text-center text-white-50 py-5">
            <i class="bi bi-server display-4 mb-3 d-block"></i>
            <p>Nema servera za prikaz</p>
        </div>
        <?php endif; ?>
    </div>
</div>

<script>
function toggleServer(node) {
    node.classList.toggle('expanded');
}

function expandAll() {
    document.querySelectorAll('.server-node').forEach(n => n.classList.add('expanded'));
}

function collapseAll() {
    document.querySelectorAll('.server-node').forEach(n => n.classList.remove('expanded'));
}

// Expand first server by default
document.addEventListener('DOMContentLoaded', function() {
    const first = document.querySelector('.server-node');
    if (first) first.classList.add('expanded');
});
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
